<?php

namespace App\Controllers;

use App\Models\AsistenciaModel;
use CodeIgniter\RESTful\ResourceController;

class AsistenciasController extends ResourceController
{
    protected $format = 'json';
    protected $modelName = 'App\Models\AsistenciaModel'; // ✅ AGREGADO
    
    /**
     * GET /api/asistencias
     * Obtener todas las asistencias con filtros opcionales
     */
    public function index()
    {
        $model = new AsistenciaModel();
        
        // Filtros opcionales
        $fechaInicio = $this->request->getGet('fecha_inicio');
        $fechaFin = $this->request->getGet('fecha_fin');
        $idMateria = $this->request->getGet('id_materia');
        
        $asistencias = $model->obtenerAsistenciasConDetalles($fechaInicio, $fechaFin, $idMateria);
        
        return $this->respond([
            'success' => true,
            'message' => 'Asistencias obtenidas correctamente',
            'data' => $asistencias
        ]);
    }
    
    /**
     * ✅ NUEVO - GET /api/asistencias/fecha/{fecha}
     * Obtener asistencias por fecha específica
     */
    public function porFecha($fecha)
    {
        try {
            // Validar formato de fecha
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha)) {
                return $this->respond([
                    'success' => false,
                    'message' => 'Formato de fecha inválido. Use YYYY-MM-DD',
                    'data' => null
                ], 400);
            }

            $model = new AsistenciaModel(); // ✅ CORREGIDO
            
            // Obtener id_materia del query string (opcional)
            $idMateria = $this->request->getGet('id_materia');
            
            // Obtener asistencias
            $asistencias = $model->getAsistenciasPorFecha($fecha, $idMateria);
            
            return $this->respond([
                'success' => true,
                'data' => $asistencias,
                'message' => 'Asistencias obtenidas correctamente'
            ]);
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }
    
    /**
     * GET /api/asistencias/{id}
     */
    public function show($id = null)
    {
        $model = new AsistenciaModel();
        
        $asistencia = $model->find($id);
        
        if ($asistencia) {
            return $this->respond([
                'success' => true,
                'message' => 'Asistencia encontrada',
                'data' => $asistencia
            ]);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Asistencia no encontrada',
            'data' => null
        ], 404);
    }
    
    /**
     * POST /api/asistencias
     */
    public function create()
    {
        $model = new AsistenciaModel();
        
        $json = $this->request->getJSON();
        
        if (!$json) {
            return $this->respond([
                'success' => false,
                'message' => 'Datos inválidos',
                'data' => null
            ], 400);
        }
        
        $data = [
            'id_estudiante' => $json->id_estudiante ?? 0,
            'id_materia' => $json->id_materia ?? null,
            'fecha_asistencia' => $json->fecha_asistencia ?? date('Y-m-d'),
            'hora_registro' => $json->hora_registro ?? date('H:i:s'),
            'estado_asistencia' => $json->estado_asistencia ?? 'presente',
            'observaciones' => $json->observaciones ?? null,
            'registrado_por' => $json->registrado_por ?? null
        ];
        
        // Validaciones
        if (empty($data['id_estudiante'])) {
            return $this->respond([
                'success' => false,
                'message' => 'ID de estudiante es requerido',
                'data' => null
            ], 400);
        }
        
        if (empty($data['id_materia'])) {
            return $this->respond([
                'success' => false,
                'message' => 'ID de materia es requerido',
                'data' => null
            ], 400);
        }
        
        if (!in_array($data['estado_asistencia'], ['presente', 'ausente', 'tardanza'])) {
            return $this->respond([
                'success' => false,
                'message' => 'Estado de asistencia inválido',
                'data' => null
            ], 400);
        }
        
        // Verificar duplicados
        $existente = $model->where('id_estudiante', $data['id_estudiante'])
                           ->where('id_materia', $data['id_materia'])
                           ->where('fecha_asistencia', $data['fecha_asistencia'])
                           ->first();
        
        if ($existente) {
            return $this->respond([
                'success' => false,
                'message' => 'Ya existe un registro de asistencia para este estudiante en esta materia hoy',
                'data' => null
            ], 409);
        }
        
        if ($model->insert($data)) {
            return $this->respond([
                'success' => true,
                'message' => 'Asistencia registrada exitosamente',
                'data' => ['id' => $model->getInsertID()]
            ], 201);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al registrar asistencia',
            'data' => null
        ], 500);
    }
    
    /**
     * PUT /api/asistencias/{id}
     */
    public function update($id = null)
    {
        $model = new AsistenciaModel();
        
        $asistencia = $model->find($id);
        
        if (!$asistencia) {
            return $this->respond([
                'success' => false,
                'message' => 'Asistencia no encontrada',
                'data' => null
            ], 404);
        }
        
        $json = $this->request->getJSON();
        
        if (!$json) {
            return $this->respond([
                'success' => false,
                'message' => 'Datos inválidos',
                'data' => null
            ], 400);
        }
        
        $data = [
            'estado_asistencia' => $json->estado_asistencia ?? $asistencia['estado_asistencia'],
            'observaciones' => $json->observaciones ?? $asistencia['observaciones'],
            'hora_registro' => $json->hora_registro ?? $asistencia['hora_registro']
        ];
        
        if (!in_array($data['estado_asistencia'], ['presente', 'ausente', 'tardanza'])) {
            return $this->respond([
                'success' => false,
                'message' => 'Estado de asistencia inválido',
                'data' => null
            ], 400);
        }
        
        if ($model->update($id, $data)) {
            return $this->respond([
                'success' => true,
                'message' => 'Asistencia actualizada exitosamente',
                'data' => ['id' => $id]
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al actualizar asistencia',
            'data' => null
        ], 500);
    }
    
    /**
     * DELETE /api/asistencias/{id}
     */
    public function delete($id = null)
    {
        $model = new AsistenciaModel();
        
        $asistencia = $model->find($id);
        
        if (!$asistencia) {
            return $this->respond([
                'success' => false,
                'message' => 'Asistencia no encontrada',
                'data' => null
            ], 404);
        }
        
        if ($model->delete($id)) {
            return $this->respond([
                'success' => true,
                'message' => 'Asistencia eliminada exitosamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al eliminar asistencia',
            'data' => null
        ], 500);
    }
    
    /**
     * GET /api/asistencias/estadisticas
     */
    public function estadisticas()
    {
        $db = \Config\Database::connect();
        $idMateria = $this->request->getGet('id_materia');
        
        try {
            $totalEstudiantes = $db->table('estudiantes')
                ->where('estado', 'activo')
                ->countAllResults();
            
            $builderTotal = $db->table('asistencias');
            if ($idMateria) {
                $builderTotal->where('id_materia', $idMateria);
            }
            $totalAsistencias = $builderTotal->countAllResults();
            
            $hoy = date('Y-m-d');
            
            $builderPresentesHoy = $db->table('asistencias')
                ->where('estado_asistencia', 'presente')
                ->where('fecha_asistencia', $hoy);
            if ($idMateria) {
                $builderPresentesHoy->where('id_materia', $idMateria);
            }
            $presentesHoy = $builderPresentesHoy->countAllResults();
            
            $builderAusentesHoy = $db->table('asistencias')
                ->where('estado_asistencia', 'ausente')
                ->where('fecha_asistencia', $hoy);
            if ($idMateria) {
                $builderAusentesHoy->where('id_materia', $idMateria);
            }
            $ausentesHoy = $builderAusentesHoy->countAllResults();
            
            $builderTardanzasHoy = $db->table('asistencias')
                ->where('estado_asistencia', 'tardanza')
                ->where('fecha_asistencia', $hoy);
            if ($idMateria) {
                $builderTardanzasHoy->where('id_materia', $idMateria);
            }
            $tardanzasHoy = $builderTardanzasHoy->countAllResults();
            
            $totalHoy = $presentesHoy + $ausentesHoy + $tardanzasHoy;
            
            $builderPresentes = $db->table('asistencias')
                ->where('estado_asistencia', 'presente');
            if ($idMateria) {
                $builderPresentes->where('id_materia', $idMateria);
            }
            $totalPresentes = $builderPresentes->countAllResults();
            
            $porcentajeAsistencia = $totalAsistencias > 0 
                ? round(($totalPresentes * 100) / $totalAsistencias, 2) 
                : 0;
            
            $fechaSemana = date('Y-m-d', strtotime('-7 days'));
            $builderSemana = $db->table('asistencias')
                ->where('fecha_asistencia >=', $fechaSemana);
            if ($idMateria) {
                $builderSemana->where('id_materia', $idMateria);
            }
            $asistenciasSemana = $builderSemana->countAllResults();
            
            $inicioMes = date('Y-m-01');
            $builderMes = $db->table('asistencias')
                ->where('fecha_asistencia >=', $inicioMes);
            if ($idMateria) {
                $builderMes->where('id_materia', $idMateria);
            }
            $asistenciasMes = $builderMes->countAllResults();
            
            $builderTop = $db->table('asistencias')
                ->select('estudiantes.nombres, estudiantes.apellidos, COUNT(*) as total')
                ->join('estudiantes', 'estudiantes.id_estudiante = asistencias.id_estudiante')
                ->where('asistencias.estado_asistencia', 'presente');
            if ($idMateria) {
                $builderTop->where('asistencias.id_materia', $idMateria);
            }
            $topEstudiantes = $builderTop
                ->groupBy('asistencias.id_estudiante')
                ->orderBy('total', 'DESC')
                ->limit(5)
                ->get()
                ->getResultArray();
            
            return $this->respond([
                'success' => true,
                'message' => 'Estadísticas obtenidas correctamente',
                'data' => [
                    'total_estudiantes' => $totalEstudiantes,
                    'total_asistencias' => $totalAsistencias,
                    'porcentaje_asistencia' => $porcentajeAsistencia,
                    'asistencias_hoy' => [
                        'presentes' => $presentesHoy,
                        'ausentes' => $ausentesHoy,
                        'tardanzas' => $tardanzasHoy,
                        'total' => $totalHoy
                    ],
                    'asistencias_semana' => $asistenciasSemana,
                    'asistencias_mes' => $asistenciasMes,
                    'top_estudiantes' => $topEstudiantes
                ]
            ], 200);
            
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error al obtener estadísticas: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }

    /**
     * GET /api/asistencias/historial/:id
     */
    public function historialEstudiante($idEstudiante = null)
    {
        $db = \Config\Database::connect();
        
        if (!$idEstudiante) {
            return $this->respond([
                'success' => false,
                'message' => 'ID de estudiante requerido',
                'data' => null
            ], 400);
        }
        
        try {
            $estudiante = $db->table('estudiantes')
                ->where('id_estudiante', $idEstudiante)
                ->get()
                ->getRowArray();
            
            if (!$estudiante) {
                return $this->respond([
                    'success' => false,
                    'message' => 'Estudiante no encontrado',
                    'data' => null
                ], 404);
            }
            
            $fechaInicio = $this->request->getGet('fecha_inicio');
            $fechaFin = $this->request->getGet('fecha_fin');
            $idMateria = $this->request->getGet('id_materia');
            
            $builder = $db->table('asistencias')
                ->select('asistencias.*, materias.nombre_materia')
                ->join('materias', 'materias.id_materia = asistencias.id_materia', 'left')
                ->where('asistencias.id_estudiante', $idEstudiante)
                ->orderBy('asistencias.fecha_asistencia', 'DESC');
            
            if ($fechaInicio) {
                $builder->where('asistencias.fecha_asistencia >=', $fechaInicio);
            }
            
            if ($fechaFin) {
                $builder->where('asistencias.fecha_asistencia <=', $fechaFin);
            }
            
            if ($idMateria) {
                $builder->where('asistencias.id_materia', $idMateria);
            }
            
            $asistencias = $builder->get()->getResultArray();
            
            $totalAsistencias = count($asistencias);
            
            $presentes = 0;
            $ausentes = 0;
            $tardanzas = 0;
            
            foreach ($asistencias as $asistencia) {
                switch ($asistencia['estado_asistencia']) {
                    case 'presente':
                        $presentes++;
                        break;
                    case 'ausente':
                        $ausentes++;
                        break;
                    case 'tardanza':
                        $tardanzas++;
                        break;
                }
            }
            
            $porcentajeAsistencia = $totalAsistencias > 0 
                ? round(($presentes / $totalAsistencias) * 100, 2) 
                : 0;
            
            return $this->respond([
                'success' => true,
                'message' => 'Historial obtenido correctamente',
                'data' => [
                    'estudiante' => [
                        'id_estudiante' => $estudiante['id_estudiante'],
                        'codigo_estudiante' => $estudiante['codigo_estudiante'],
                        'nombres' => $estudiante['nombres'],
                        'apellidos' => $estudiante['apellidos'],
                        'email' => $estudiante['email'],
                        'telefono' => $estudiante['telefono']
                    ],
                    'estadisticas' => [
                        'total_asistencias' => $totalAsistencias,
                        'presentes' => $presentes,
                        'ausentes' => $ausentes,
                        'tardanzas' => $tardanzas,
                        'porcentaje_asistencia' => $porcentajeAsistencia
                    ],
                    'asistencias' => $asistencias
                ]
            ], 200);
            
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error al obtener historial: ' . $e->getMessage(),
                'data' => null
            ], 500);
        }
    }
}

