<?php

namespace App\Controllers;

use App\Models\UsuarioModel;
use App\Helpers\JwtHelper;
use CodeIgniter\RESTful\ResourceController;

class AuthController extends ResourceController
{
    protected $format = 'json';
    
    /**
     * POST /api/login
     * Iniciar sesión y devolver un token JWT.
     */
    public function login()
    {
        $model = new UsuarioModel();
        
        $json = $this->request->getJSON();
        
        if (!$json || !isset($json->username) || !isset($json->password)) {
            return $this->respond([
                'success' => false,
                'message' => 'Usuario y contraseña son requeridos',
            ], 400);
        }
        
        $username = $json->username;
        $password = $json->password;
        
        $usuario = $model->verificarCredenciales($username, $password);
        
        if ($usuario) {
            // Prepara el payload con los datos del usuario para el token
            $payload = [
                'id_usuario' => $usuario['id_usuario'],
                'username' => $usuario['username'],
                'tipo_usuario' => $usuario['tipo_usuario'],
            ];

            // Genera el token JWT
            $token = JwtHelper::generateToken($payload);

            log_message('debug', 'Login exitoso para usuario: ' . $usuario['username']);
            
            // Prepara la respuesta sin datos sensibles como la contraseña
            $responseData = [
                'id_usuario' => $usuario['id_usuario'],
                'username' => $usuario['username'],
                'email' => $usuario['email'],
                'tipo_usuario' => $usuario['tipo_usuario'],
                'token' => $token
            ];
            
            return $this->respond([
                'success' => true,
                'message' => 'Login exitoso',
                'data' => $responseData
            ]);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Credenciales incorrectas',
        ], 401);
    }
    
    /**
     * POST /api/register
     * Registrar nuevo usuario
     */
    public function register()
    {
        $model = new UsuarioModel();
        
        $json = $this->request->getJSON();
        
        if (!$json) {
            return $this->respond([
                'success' => false,
                'message' => 'Datos inválidos',
                'data' => null
            ], 400);
        }
        
        $data = [
            'username' => $json->username ?? '',
            'email' => $json->email ?? '',
            'password' => $json->password ?? '',
            'tipo_usuario' => $json->tipo_usuario ?? 'docente'
        ];
        
        // Validaciones
        if (empty($data['username']) || empty($data['email']) || empty($data['password'])) {
            return $this->respond([
                'success' => false,
                'message' => 'Todos los campos son requeridos',
                'data' => null
            ], 400);
        }
        
        if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            return $this->respond([
                'success' => false,
                'message' => 'Email inválido',
                'data' => null
            ], 400);
        }
        
        // Verificar si ya existe
        if ($model->where('username', $data['username'])->first()) {
            return $this->respond([
                'success' => false,
                'message' => 'El usuario ya existe',
                'data' => null
            ], 409);
        }
        
        if ($model->where('email', $data['email'])->first()) {
            return $this->respond([
                'success' => false,
                'message' => 'El email ya está registrado',
                'data' => null
            ], 409);
        }
        
        // Insertar usuario
        if ($model->insert($data)) {
            $usuario = $model->find($model->getInsertID());
            unset($usuario['password']);
            
            return $this->respond([
                'success' => true,
                'message' => 'Usuario registrado exitosamente',
                'data' => $usuario
            ], 201);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al registrar usuario',
            'data' => null
        ], 500);
    }

    /**
     * GET /api/perfil
     * Obtener datos del usuario autenticado
     */
    public function perfil()
    {
        // ✅ Obtener el usuario autenticado del filtro
        $usuario = $this->request->usuario;
        
        log_message('debug', '✅ Perfil - Usuario autenticado: ' . $usuario['username']);
        
        // No enviar la contraseña ni el token
        unset($usuario['password']);
        unset($usuario['token']);
        
        return $this->respond([
            'success' => true,
            'message' => 'Perfil obtenido correctamente',
            'data' => $usuario
        ], 200);
    }

    /**
     * PUT /api/perfil
     * Actualizar datos del perfil
     */
    public function actualizarPerfil()
    {
        // ✅ Obtener el usuario autenticado del filtro
        $usuario = $this->request->usuario;
        
        log_message('debug', '✅ Actualizar perfil - Usuario: ' . $usuario['username']);
        
        $json = $this->request->getJSON();
        
        if (!$json) {
            return $this->respond([
                'success' => false,
                'message' => 'Datos inválidos',
                'data' => null
            ], 400);
        }
        
        $model = new UsuarioModel();
        
        // ✅ Preparar datos a actualizar
        $data = [];
        
        if (isset($json->telefono)) {
            $data['telefono'] = $json->telefono;
        }
        
        if (isset($json->especialidad)) {
            $data['especialidad'] = $json->especialidad;
        }
        
        if (isset($json->email)) {
            // Validar email único si cambió
            if ($json->email != $usuario['email']) {
                if (!filter_var($json->email, FILTER_VALIDATE_EMAIL)) {
                    return $this->respond([
                        'success' => false,
                        'message' => 'Email inválido',
                        'data' => null
                    ], 400);
                }
                
                $emailExiste = $model->where('email', $json->email)
                                    ->where('id_usuario !=', $usuario['id_usuario'])
                                    ->first();
                
                if ($emailExiste) {
                    return $this->respond([
                        'success' => false,
                        'message' => 'El email ya está registrado',
                        'data' => null
                    ], 409);
                }
                
                $data['email'] = $json->email;
            }
        }
        
        if (empty($data)) {
            return $this->respond([
                'success' => false,
                'message' => 'No hay datos para actualizar',
                'data' => null
            ], 400);
        }
        
        log_message('debug', '📝 Actualizando perfil: ' . json_encode($data));
        
        // ✅ Actualizar en la base de datos
        if ($model->update($usuario['id_usuario'], $data)) {
            // Obtener usuario actualizado
            $usuarioActualizado = $model->find($usuario['id_usuario']);
            unset($usuarioActualizado['password']);
            unset($usuarioActualizado['token']);
            
            log_message('debug', '✅ Perfil actualizado correctamente');
            
            return $this->respond([
                'success' => true,
                'message' => 'Perfil actualizado correctamente',
                'data' => $usuarioActualizado
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al actualizar perfil',
            'data' => null
        ], 500);
    }

    /**
     * PUT /api/cambiar-password
     * Cambiar contraseña
     */
    public function cambiarPassword()
    {
        // ✅ Obtener el usuario autenticado del filtro
        $usuario = $this->request->usuario;
        
        $json = $this->request->getJSON();
        
        if (!isset($json->password_actual) || !isset($json->password_nueva)) {
            return $this->respond([
                'success' => false,
                'message' => 'Contraseña actual y nueva son requeridas',
                'data' => null
            ], 400);
        }
        
        // Verificar contraseña actual
        if (!password_verify($json->password_actual, $usuario['password'])) {
            return $this->respond([
                'success' => false,
                'message' => 'La contraseña actual es incorrecta',
                'data' => null
            ], 401);
        }
        
        // Validar nueva contraseña
        if (strlen($json->password_nueva) < 6) {
            return $this->respond([
                'success' => false,
                'message' => 'La nueva contraseña debe tener al menos 6 caracteres',
                'data' => null
            ], 400);
        }
        
        $model = new UsuarioModel();
        
        // Actualizar contraseña
        $data = [
            'password' => password_hash($json->password_nueva, PASSWORD_DEFAULT)
        ];
        
        if ($model->update($usuario['id_usuario'], $data)) {
            log_message('debug', '✅ Contraseña actualizada para usuario: ' . $usuario['username']);
            
            return $this->respond([
                'success' => true,
                'message' => 'Contraseña actualizada correctamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al cambiar contraseña',
            'data' => null
        ], 500);
    }

    /**
     * DELETE /api/cuenta
     * Eliminar cuenta
     */
    public function eliminarCuenta()
    {
        // ✅ Obtener el usuario autenticado del filtro
        $usuario = $this->request->usuario;
        
        $json = $this->request->getJSON();
        
        if (!isset($json->password)) {
            return $this->respond([
                'success' => false,
                'message' => 'Contraseña requerida para eliminar cuenta',
                'data' => null
            ], 400);
        }
        
        // Verificar contraseña
        if (!password_verify($json->password, $usuario['password'])) {
            return $this->respond([
                'success' => false,
                'message' => 'Contraseña incorrecta',
                'data' => null
            ], 401);
        }
        
        $model = new UsuarioModel();
        
        // Eliminar cuenta
        if ($model->delete($usuario['id_usuario'])) {
            log_message('info', '🗑️ Cuenta eliminada - Usuario: ' . $usuario['username']);
            
            return $this->respond([
                'success' => true,
                'message' => 'Cuenta eliminada correctamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al eliminar cuenta',
            'data' => null
        ], 500);
    }
}
