<?php

namespace App\Controllers;

use App\Models\MateriaModel;
use CodeIgniter\RESTful\ResourceController;

class MateriasController extends ResourceController
{
    protected $modelName = 'App\Models\MateriaModel';
    protected $format = 'json';
    
    /**
     * GET /api/materias
     * Listar todas las materias del docente
     */
    public function index()
    {
        $idDocente = $this->request->getHeaderLine('X-User-ID') ?: 1;
        
        $materiaModel = new MateriaModel();
        $materias = $materiaModel->getMateriasConEstudiantes($idDocente);
        
        return $this->respond([
            'success' => true,
            'message' => 'Materias obtenidas correctamente',
            'data' => $materias
        ], 200);
    }
    
    /**
     * GET /api/materias/{id}
     * Obtener una materia específica
     */
    public function show($id = null)
    {
        $materiaModel = new MateriaModel();
        $materia = $materiaModel->find($id);
        
        if (!$materia) {
            return $this->respond([
                'success' => false,
                'message' => 'Materia no encontrada',
                'data' => null
            ], 404);
        }
        
        return $this->respond([
            'success' => true,
            'message' => 'Materia obtenida correctamente',
            'data' => $materia
        ], 200);
    }
    
    /**
     * POST /api/materias
     * Crear nueva materia
     */
    public function create()
    {
        $idDocente = $this->request->getHeaderLine('X-User-ID') ?: 1;
        
        $json = $this->request->getJSON();
        
        $data = [
            'nombre_materia' => $json->nombre_materia ?? '',
            'descripcion' => $json->descripcion ?? null,
            'grado' => $json->grado ?? null,
            'seccion' => $json->seccion ?? null,
            'id_docente' => $idDocente
        ];
        
        // Validaciones
        if (empty($data['nombre_materia'])) {
            return $this->respond([
                'success' => false,
                'message' => 'El nombre de la materia es requerido',
                'data' => null
            ], 400);
        }
        
        $materiaModel = new MateriaModel();
        
        if ($materiaModel->insert($data)) {
            $materia = $materiaModel->find($materiaModel->getInsertID());
            
            return $this->respond([
                'success' => true,
                'message' => 'Materia creada correctamente',
                'data' => $materia
            ], 201);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al crear materia',
            'data' => null
        ], 500);
    }
    
    /**
     * PUT /api/materias/{id}
     * Actualizar materia
     */
    public function update($id = null)
    {
        $materiaModel = new MateriaModel();
        $materia = $materiaModel->find($id);
        
        if (!$materia) {
            return $this->respond([
                'success' => false,
                'message' => 'Materia no encontrada',
                'data' => null
            ], 404);
        }
        
        $json = $this->request->getJSON();
        
        $data = [
            'nombre_materia' => $json->nombre_materia ?? $materia['nombre_materia'],
            'descripcion' => $json->descripcion ?? $materia['descripcion'],
            'grado' => $json->grado ?? $materia['grado'],
            'seccion' => $json->seccion ?? $materia['seccion']
        ];
        
        if ($materiaModel->update($id, $data)) {
            $materiaActualizada = $materiaModel->find($id);
            
            return $this->respond([
                'success' => true,
                'message' => 'Materia actualizada correctamente',
                'data' => $materiaActualizada
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al actualizar materia',
            'data' => null
        ], 500);
    }
    
    /**
     * DELETE /api/materias/{id}
     * Eliminar materia
     */
    public function delete($id = null)
    {
        $materiaModel = new MateriaModel();
        $materia = $materiaModel->find($id);
        
        if (!$materia) {
            return $this->respond([
                'success' => false,
                'message' => 'Materia no encontrada',
                'data' => null
            ], 404);
        }
        
        if ($materiaModel->delete($id)) {
            return $this->respond([
                'success' => true,
                'message' => 'Materia eliminada correctamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al eliminar materia',
            'data' => null
        ], 500);
    }
    
    /**
     * POST /api/materias/{id}/estudiantes
     * Asignar estudiante a materia
     */
    public function asignarEstudiante($id = null)
    {
        $json = $this->request->getJSON();
        
        if (!isset($json->id_estudiante)) {
            return $this->respond([
                'success' => false,
                'message' => 'ID de estudiante requerido',
                'data' => null
            ], 400);
        }
        
        $materiaModel = new MateriaModel();
        
        if ($materiaModel->asignarEstudiante($id, $json->id_estudiante)) {
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante asignado correctamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'El estudiante ya está asignado a esta materia',
            'data' => null
        ], 409);
    }
    
    /**
     * DELETE /api/materias/{id}/estudiantes/{idEstudiante}
     * Desasignar estudiante de materia
     */
    public function desasignarEstudiante($id = null, $idEstudiante = null)
    {
        $materiaModel = new MateriaModel();
        
        if ($materiaModel->desasignarEstudiante($id, $idEstudiante)) {
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante desasignado correctamente',
                'data' => null
            ], 200);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al desasignar estudiante',
            'data' => null
        ], 500);
    }
    
    /**
     * GET /api/materias/{id}/estudiantes
     * Obtener estudiantes de una materia
     */
    public function getEstudiantes($id = null)
    {
        $materiaModel = new MateriaModel();
        $estudiantes = $materiaModel->getEstudiantesDeMateria($id);
        
        return $this->respond([
            'success' => true,
            'message' => 'Estudiantes obtenidos correctamente',
            'data' => $estudiantes
        ], 200);
    }
}
