<?php

namespace App\Models;

use CodeIgniter\Model;

class UsuarioModel extends Model
{
    protected $table = 'usuarios';
    protected $primaryKey = 'id_usuario';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'username', 
        'email', 
        'password', 
        'tipo_usuario', 
        'estado',
        'telefono',
        'especialidad'
        // El campo 'token' ya no se usa y se puede omitir.
    ];
    
    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'fecha_registro';
    protected $updatedField = ''; // No hay campo updated_at en la tabla

    // Validation
    protected $validationRules = [
        'username' => 'required|min_length[3]|max_length[50]|is_unique[usuarios.username,id_usuario,{id_usuario}]',
        'email'    => 'required|max_length[100]|valid_email|is_unique[usuarios.email,id_usuario,{id_usuario}]',
        'password' => 'permit_empty|min_length[6]', // `permit_empty` para que no sea requerido en actualizaciones
    ];

    protected $validationMessages = [
        'username' => [
            'required'   => 'El nombre de usuario es obligatorio.',
            'min_length' => 'El nombre de usuario debe tener al menos 3 caracteres.',
            'max_length' => 'El nombre de usuario no puede exceder los 50 caracteres.',
            'is_unique'  => 'El nombre de usuario ya está en uso. Por favor, elige otro.',
        ],
        'email' => [
            'required'    => 'El email es obligatorio.',
            'max_length'  => 'El email no puede exceder los 100 caracteres.',
            'valid_email' => 'Por favor, introduce una dirección de email válida.',
            'is_unique'   => 'Esta dirección de email ya está registrada. Por favor, usa otra.',
        ],
        'password' => [
            'min_length' => 'La contraseña debe tener al menos 6 caracteres.',
        ],
    ];

    // Callbacks
    protected $beforeInsert = ['hashPassword'];
    protected $beforeUpdate = ['hashPassword'];
    
    /**
     * Hashea la contraseña antes de guardarla.
     */
    protected function hashPassword(array $data)
    {
        if (isset($data['data']['password'])) {
            // Solo hashear si la contraseña no está vacía
            if (!empty($data['data']['password'])) {
                $data['data']['password'] = password_hash($data['data']['password'], PASSWORD_DEFAULT);
            } else {
                // Si la contraseña está vacía, la eliminamos para no sobreescribir la existente con un hash vacío
                unset($data['data']['password']);
            }
        }
        return $data;
    }
    
    /**
     * Verificar credenciales de login
     */
    public function verificarCredenciales($username, $password)
    {
        $usuario = $this->where('username', $username)
                         ->orWhere('email', $username)
                         ->first();
        
        if ($usuario && password_verify($password, $usuario['password'])) {
            return $usuario;
        }
        
        return null;
    }
}
