<?php

namespace App\Helpers;

// Importa las clases de la librería JWT que se instalará con Composer.
// Este código no funcionará hasta que se ejecute: composer require firebase/php-jwt
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class JwtHelper
{
    /**
     * Genera un token JWT para un usuario.
     *
     * @param array $data Los datos del usuario para incluir en el payload.
     * @return string El token JWT generado.
     */
    public static function generateToken(array $data): string
    {
        $secretKey = getenv('jwt.secret');
        $issuedAt = time();
        $expirationTime = $issuedAt + (60 * 60 * 24); // Token válido por 24 horas

        $payload = [
            'iat'  => $issuedAt,
            'exp'  => $expirationTime,
            'data' => $data,
        ];

        // Codifica el payload para generar el token.
        // Se asume que el algoritmo es HS256.
        return JWT::encode($payload, $secretKey, 'HS256');
    }

    /**
     * Decodifica y valida un token JWT.
     *
     * @param string $token El token JWT a validar.
     * @return object|null El payload decodificado si el token es válido, o null si no lo es.
     */
    public static function decodeToken(string $token)
    {
        $secretKey = getenv('jwt.secret');

        if (!$secretKey) {
            // No se puede validar si no hay clave secreta.
            return null;
        }

        try {
            // Intenta decodificar el token. La librería se encarga de verificar
            // la firma y la fecha de expiración.
            $decoded = JWT::decode($token, new Key($secretKey, 'HS256'));
            return $decoded;
        } catch (\Exception $e) {
            // El token es inválido (expirado, firma incorrecta, etc.).
            return null;
        }
    }
}
